#!/bin/bash

# Función para generar una contraseña segura
generate_password() {
    openssl rand -base64 8 | tr -cd 'a-zA-Z0-9'
}

# Generar la clave privada
PRIVATE_KEY=$(openssl genpkey -algorithm RSA -outform PEM -pkeyopt rsa_keygen_bits:2048)

# Generar la clave pública a partir de la clave privada
PUBLIC_KEY=$(echo "$PRIVATE_KEY" | openssl rsa -pubout)

# Función para preguntar y validar una entrada
prompt_and_validate() {
    local prompt_message="$1"
    local result_var="$2"
    while true; do
        read -p "$prompt_message" input
        if validate_input "$input"; then
            eval "$result_var='$input'"
            break
        fi
    done
}
# Función para validar que una entrada no esté vacía
validate_input() {
    if [ -z "$1" ]; then
        echo "Entrada inválida. Por favor, inténtelo de nuevo."
        return 1
    fi
    return 0
}

# Preguntar si la base de datos es local o externa
read -p "¿La base de datos se creará de forma local o es conexión externa? (local/external): " database_choice

if [ "$database_choice" = "local" ]; then
    db_user="smartgrow"
    db_name="smartgrow"
    db_password==$(generate_password)
else
    while true; do
        read -p "Ingrese el nombre de la base de datos seguido de la rama (si existe, ejemplo db_name:main): " db_name
        validate_input "$db_name" && break
    done
    while true; do
        read -p "Ingrese el nombre de usuario de la base de datos: " db_user
        validate_input "$db_user" && break
    done
    while true; do
        read -p "Ingrese la contraseña de la base de datos: " db_password
        validate_input "$db_password" && break
    done
    # Preguntar la dirección del servidor y el puerto de la base de datos
    prompt_and_validate "Ingrese la dirección del servidor de la base de datos (SMART_GROW_DB_HOST): " db_host
    prompt_and_validate "Ingrese el puerto de la base de datos (SMART_GROW_DB_PORT): " db_port
    # Generar el string de conexión para PostgreSQL
    db_connection_string="postgresql://$db_user:$db_password@$db_host:$db_port/$db_name?ssl=true"
fi

encryption_key=$(openssl rand -base64 32)

# Preguntar la URL base de la aplicación
prompt_and_validate "Ingrese la URL base del servidor: " base_url
prompt_and_validate "Ingrese la URL base de la aplicación Web: " base_front_url
base_front_domain=$(echo "$base_front_url" | sed -E 's~https?://([^/]+).*~\1~')

# Preguntar las credenciales del servicio de S3
prompt_and_validate "Ingrese el nombre del Bucket S3: " s3_bucket_name
prompt_and_validate "Ingrese la URL del Bucket S3: " s3_bucket_url
prompt_and_validate "Ingrese el Access Key ID del Bucket S3: " s3_bucket_access_key_id
prompt_and_validate "Ingrese la Secret Access Key del Bucket S3: " s3_bucket_secret_access_key

# Generar una clave secreta para JWT
SMART_GROW_JWT_DOWNLOAD_SECRET=$(openssl rand -base64 32)

# Preguntar las variables SMTP
prompt_and_validate "Ingrese el nombre (solo nombre) del usuario administrador: " admin_user
prompt_and_validate "Ingrese los apellidos del usuario administrador: " admin_last_name
prompt_and_validate "Ingrese el correo del usuario administrador: " admin_email
prompt_and_validate "Ingrese el nombre de usuario del servicio SMTP (SMTP_USER): " smtp_user
prompt_and_validate "Ingrese la contraseña del servicio SMTP (SMTP_PASSWORD): " smtp_password
prompt_and_validate "Ingrese el host del servicio SMTP (SMTP_HOST): " smtp_host
prompt_and_validate "Ingrese el puerto del servicio SMTP (SMTP_PORT): " smtp_port
prompt_and_validate "Utilizar seguridad del servicio SMTP (true/false) (SMTP_SECURE): " smtp_secure

# Guardar todas las variables en un archivo .env con comillas dobles
cat <<EOF >.env
SMART_GROW_ENCRYPTION_KEY="$encryption_key"
ADMIN_USER="$admin_user"
ADMIN_LAST_NAME="$admin_last_name"
ADMIN_EMAIL="$admin_email"
BASE_URL="$base_url"
BASE_FRONT_URL="$base_front_url"
BASE_FRONT_DOMAIN="$base_front_domain"
SMART_GROW_DB_USER="$db_user"
SMART_GROW_DB_NAME="$db_name"
SMART_GROW_DB_PASSWORD="$db_password"
SMART_GROW_DB_HOST="$db_host"
SMART_GROW_DB_PORT="$db_port"
SMART_GROW_DB_URL="$db_connection_string"
SMART_GROW_JWT_DOWNLOAD_SECRET="$SMART_GROW_JWT_DOWNLOAD_SECRET"
S3_BUCKET_NAME="$s3_bucket_name"
S3_BUCKET_URL="$s3_bucket_url"
S3_BUCKET_ACCESS_KEY_ID="$s3_bucket_access_key_id"
S3_BUCKET_SECRET_ACCESS_KEY="$s3_bucket_secret_access_key"
SMTP_USER="$smtp_user"
SMTP_PASSWORD="$smtp_password"
SMTP_HOST="$smtp_host"
SMTP_PORT="$smtp_port"
SMTP_SECURE="$smtp_secure"
ENABLE_RATE_LIMITER="true"
LIMITER_RATE_PER_MINUTE=100
HTTP_PORT=3001
ENVIRONMENT="PRODUCTION"
ENABLE_LOGGING="false"
LOGGING_DSN=""
ENABLE_CORS="true"
PROXY_HOPS=2
INITIAL_LICENSE_KEY=""
PUBLIC_RSA_KEY="$PUBLIC_KEY"
PRIVATE_RSA_KEY="$PRIVATE_KEY"
EOF

echo "Se han guardado las variables en el archivo .env"

# Crear red de Docker si no existe
docker network create proxy || echo "La red proxy ya existe"

if [ "$database_choice" = "local" ]; then
    docker network create database || echo "La red database ya existe"
    echo "Instalando la base de datos local..."
    # Levantar el servicio de la base de datos
    docker compose -f docker-compose-db.yml up -d

    # Esperar a que la base de datos esté lista (puedes ajustar el tiempo de espera según tus necesidades)
    echo "Esperando a que la base de datos se inicialice..."
    sleep 50

    # Levantar el servicio de la aplicación
    docker compose up -d
else
    docker compose -f docker-compose-external-db.yml up -d
    mv docker-compose-external-db.yml docker-compose.yml
fi

echo "Se ha instalado Smart Grow Server de manera correcta"
